<?php
/**
 * Core: Plugin Bootstrap
 *
 * @package     AffiliateWP Pushover
 * @subpackage  Core
 * @copyright   Copyright (c) 2021, Sandhills Development, LLC
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       1.1
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Main plugin bootstrap.
 *
 * @since 1.0.0
 */
final class AffiliateWP_Pushover {

	/** Singleton *************************************************************/

	/**
	 * Holds the instance
	 *
	 * @var AffiliateWP_Pushover The one true AffiliateWP_Pushover
	 * @since 1.0
	 */
	private static $instance;

	/**
	 * The version number of AffiliateWP Pushover
	 *
	 * @since 1.0
	 * @var   string
	 */
	private $version = '1.2';

	/**
	 * Main plugin file.
	 *
	 * @since 1.1
	 * @var   string
	 */
	private $file = '';

	/**
	 * Main AffiliateWP_Pushover Instance
	 *
	 * Insures that only one instance of AffiliateWP_Pushover exists in memory at any one
	 * time. Also prevents needing to define globals all over the place.
	 *
	 * @since 1.0
	 * @static
	 * @staticvar array $instance
	 *
	 * @param string $file Main plugin file.
	 * @return The one true AffiliateWP_Pushover
	 */
	public static function instance( $file = null ) {
		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof AffiliateWP_Pushover ) ) {
			self::$instance = new AffiliateWP_Pushover;
			self::$instance->file = $file;
			self::$instance->setup_constants();
			self::$instance->load_textdomain();
			self::$instance->includes();
			self::$instance->init();
		}
		return self::$instance;
	}

	/**
	 * Throw error on object clone
	 *
	 * The whole idea of the singleton design pattern is that there is a single
	 * object therefore, we don't want the object to be cloned.
	 *
	 * @since 1.0
	 * @access protected
	 * @return void
	 */
	public function __clone() {
		// Cloning instances of the class is forbidden.
		_doing_it_wrong( __FUNCTION__, __( 'Cheatin&#8217; huh?', 'affiliate-wp-pushover' ), '1.0' );
	}

	/**
	 * Disable unserializing of the class
	 *
	 * @since 1.0
	 * @access protected
	 * @return void
	 */
	public function __wakeup() {
		// Unserializing instances of the class is forbidden.
		_doing_it_wrong( __FUNCTION__, __( 'Cheatin&#8217; huh?', 'affiliate-wp-pushover' ), '1.0' );
	}

	/**
	 * Setup plugin constants.
	 *
	 * @access private
	 * @since 1.1
	 * @return void
	 */
	private function setup_constants() {
		// Plugin version.
		if ( ! defined( 'AFFWP_WPO_VERSION' ) ) {
			define( 'AFFWP_WPO_VERSION', $this->version );
		}

		// Plugin Folder Path.
		if ( ! defined( 'AFFWP_WPO_PLUGIN_DIR' ) ) {
			define( 'AFFWP_WPO_PLUGIN_DIR', plugin_dir_path( $this->file ) );
		}

		// Plugin Folder URL.
		if ( ! defined( 'AFFWP_WPO_PLUGIN_URL' ) ) {
			define( 'AFFWP_WPO_PLUGIN_URL', plugin_dir_url( $this->file ) );
		}

		// Plugin Root File.
		if ( ! defined( 'AFFWP_WPO_PLUGIN_FILE' ) ) {
			define( 'AFFWP_WPO_PLUGIN_FILE', $this->file );
		}
	}

	/**
	 * Loads the plugin language files
	 *
	 * @access public
	 * @since 1.0
	 * @return void
	 */
	public function load_textdomain() {

		// Set filter for plugin's languages directory.
		$lang_dir = dirname( plugin_basename( $this->file ) ) . '/languages/';
		$lang_dir = apply_filters( 'aff_wp_languages_directory', $lang_dir );

		// Traditional WordPress plugin locale filter.
		$locale = apply_filters( 'plugin_locale', get_locale(), 'affiliate-wp-pushover' );
		$mofile = sprintf( '%1$s-%2$s.mo', 'affiliate-wp-pushover', $locale );

		// Setup paths to current locale file.
		$mofile_local  = $lang_dir . $mofile;
		$mofile_global = WP_LANG_DIR . '/affiliate-wp-pushover/' . $mofile;

		if ( file_exists( $mofile_global ) ) {
			// Look in global /wp-content/languages/affiliate-wp-pushover/ folder.
			load_textdomain( 'affiliate-wp-pushover', $mofile_global );
		} elseif ( file_exists( $mofile_local ) ) {
			// Look in local /wp-content/plugins/affiliate-wp-pushover/languages/ folder.
			load_textdomain( 'affiliate-wp-pushover', $mofile_local );
		} else {
			// Load the default language files.
			load_plugin_textdomain( 'affiliate-wp-pushover', false, $lang_dir );
		}
	}

	/**
	 * Include required files
	 *
	 * @access private
	 * @since 1.0
	 * @return void
	 */
	private function includes() {

		require_once AFFWP_WPO_PLUGIN_DIR . 'includes/hooks.php';

		if ( is_admin() ) {
			require_once AFFWP_WPO_PLUGIN_DIR . 'includes/admin/settings.php';
		}

	}

	/**
	 * Any filters needed at init
	 *
	 * @access private
	 * @since 1.0
	 * @return void
	 */
	private function init() {

		if ( is_admin() ) {
			self::$instance->updater();
		}

	}

	/**
	 * Plugin Updater
	 *
	 * @access private
	 * @since 1.0
	 * @return void
	 */
	public function updater() {
		if ( class_exists( 'AffWP_AddOn_Updater' ) ) {
			$updater = new AffWP_AddOn_Updater( 13199, $this->file, $this->version );
		}
	}

}

/**
 * The main function responsible for returning the one true AffiliateWP_Pushover
 * Instance to functions everywhere.
 *
 * Use this function like you would a global variable, except without needing
 * to declare the global.
 *
 * @since 1.0
 *
 * @return object The one true AffiliateWP_Pushover Instance
 */
function affiliate_wp_pushover() {
	return AffiliateWP_Pushover::instance();
}
